#*******************************************************************************
# Author: Cesar D. Jimenez-Rodriguez
# Water Resources Section, Delft University of Technology, Stevinweg 1, 2628 CN
# Delft, the Netherlands
# Tecnológico de Costa Rica, Escuela de Ingeniería Forestal, 159-7050, 
# Cartago, Costa Rica
# E-mail: cdjimenezcr@gmail.com
#*******************************************************************************

#-Script Description--------------------------------------------------------------------------------
# This script aims to analyzed the atmospheric data collected at the MRI-plot in 
# La Selva Biological Station Costa Rica. This data allowed the description of 
# the underlying processes that allowed the formation of visible vapor plumes 
# during rain events.

#-Required Packages---------------------------------------------------------------------------------
library(devtools)
library(animation)
library(akima)
library(reshape2)
library(caTools)
library(gstat)
library(lubridate)
library(agricolae)
library(lsmeans)
library(gplots)
library(foreign)
library(chron)
library(stringr)
library(stringi)
library(zoo) # Important for filling in the gaps and placing continuosly the values from the previos non empty section
library(dplyr)
library(splitstackshape)
library(data.table)
library(dplyr)
library(dtplyr)
library(plyr) #use to replace data with the command: revalue
library(plotrix)
library(zoom)
library(fields)# for colo pallette gradient
library(SDMTools)

#-User settings-----------------------------------------------------------------
#-Home Directories
a <- "C:/" # Home directory where data is store
b <- "C:/" # Home directory to store final figures

setwd(a)

#Evaporation
ELS <- read.csv("./evaporacionLS.txt",sep=",")

#Plumes Record
rcrd <- read.csv("./plumes_record.txt",sep="\t")
rcrd <- na.omit(rcrd)
rcrd$DOY <- strptime(rcrd$date, format = "%Y%m%d")$yday+1 # Julian day
rcrd$doy <- rcrd$DOY+(rcrd$hour/24)+(rcrd$minute/1440)
rcrd[c("DOY","hour","minute")] <- NULL

#OTS Meteo Station
ots <- data.frame(read.csv("./Datos_Oficiales/OTS-METEO-LS-detailed.txt", sep = "\t"))
ots[c("MaxAirTemp","MinAirTemp","Time","TimeofMaxAirTemp","TimeofMinAirTemp", "LeafWTDry", "LeafWTWet", "LeafWTCont", "StdpsactiveRad","SdsolarRad","MaxpsactiveRad","MinpsactiveRad","Sdwinddirection","Scalarmeanhr.windspeed","MinsolarRad","MaxsolarRad")]<-NULL
colnames(ots) <- c("Date","temp","rh","rf","wspeed","wdir","press","rad","par")
ots <- ots[order(ots$Date),]
ots$DOY <- strptime(ots$Date, format = "%m/%d/%Y %H:%M")$yday+1 # Julian day
ots$hr <- as.numeric(hours(strptime(ots$Date,format = "%m/%d/%Y %H:%M")))
ots$min <- as.numeric(minutes(strptime(ots$Date,format = "%m/%d/%Y %H:%M")))
ots$doy <- ots$DOY+(ots$hr/24)+(ots$min/1440)
ots$mes <- as.numeric(month(strptime(ots$Date,format = "%m/%d/%Y %H:%M")))
ots$es <- 0.6108*exp((17.27*ots$temp)/(ots$temp+237.3))
ots$ea <- ots$es*ots$rh/100
ots$m <- 4098*(ots$es)/(ots$temp+237.3)^2
ots$vpd <- ifelse(ots$es-ots$ea > 0,ots$es-ots$ea,0)
ots$E <- ots$vpd*0.009807

#Temperatura del Suelo####
ts <- read.csv("./Datos_Torre/tsuelo_compilado.txt", sep = "\t")
colnames(ts) <- c("date","t5","t6","t7","t8")
ts$s5cm <- (ts$t5+ts$t7)/2; ts$s10cm <- (ts$t6+ts$t8)/2
ts[c("cntrl","t5","t6","t7","t8")] <- NULL
ts$DOY <- strptime(ts$date, format = "%d-%m-%Y %H:%M")$yday+1 # Julian day
ts$doy <- ts$DOY+(as.numeric(hour(strptime(ts$date,format = "%d-%m-%Y %H:%M")))/24)+(as.numeric(minute(strptime(ts$date,format = "%d-%m-%Y %H:%M")))/1440)
ts[c("date")] <- NULL
ts <- ts[order(ts$doy),]
ts$s5cm <- rollmean(ts$s5cm, k=3, fill = NA) # mean every 15 minutes
ts$s10cm <- rollmean(ts$s10cm, k=3, fill = NA)# mean every 15 minutes

#Tower Data Set
# T1: 37m; T2: 43m, T3: 8m, T4: 1.5m
up <- read.csv("./Datos_Torre/up_compilado.txt", sep = "\t")
up$DOY <- strptime(up$date, format = "%d-%m-%Y %H:%M")$yday+1 # Julian day
up$doy <- up$DOY+(as.numeric(hour(strptime(up$date,format = "%d-%m-%Y %H:%M")))/24)+(as.numeric(minute(strptime(up$date,format = "%d-%m-%Y %H:%M")))/1440)
up <- up[order(up$doy),]

down <- read.csv("./Datos_Torre/down_compilado.txt", sep = "\t")
down$DOY <- strptime(down$date, format = "%d-%m-%Y %H:%M")$yday+1 # Julian day
down$doy <- down$DOY+(as.numeric(hour(strptime(down$date,format = "%d-%m-%Y %H:%M")))/24)+(as.numeric(minute(strptime(down$date,format = "%d-%m-%Y %H:%M")))/1440)
down <- down[order(down$doy),]
down[c("date","DOY")] <- NULL

tower <-data.frame(merge(up,down,by="doy", all = T)); rm("up","down")
tower <- tower[order(tower$doy),]
tower <- data.frame(tower)

twr <- merge(tower,ots[c("doy","press")],by="doy",all=T)
twr$press <- na.approx(twr$press)
twr <- na.omit(twr)
twr$press <- twr$press/10

# Potential Temperature at 100 kPa
# Video describing the process: https://www.youtube.com/watch?v=kRvfjHSw_IU
# Dry Air
twr$PdaT43m <- ((0.0098*43)+(273.15+twr$t2))-273.15
twr$PdaT37m <- ((0.0098*37)+(273.15+twr$t1))-273.15
twr$PdaT8m <- ((0.0098*8)+(273.15+twr$t3))-273.15
twr$PdaT2m <- ((0.0098*2)+(273.15+twr$t4))-273.15

# Moist Air
# Video describing the process: https://www.youtube.com/watch?v=f9zj_5wHUVQ
# Determine specific humidity

twr$qs43m <- 0.622*((0.6112*exp((17.67*twr$dp2)/(twr$dp2+243.49)))/twr$press)
twr$qs37m <- 0.622*((0.6112*exp((17.67*twr$dp1)/(twr$dp1+243.49)))/twr$press)
twr$qs8m <- 0.622*((0.6112*exp((17.67*twr$dp3)/(twr$dp3+243.49)))/twr$press)
twr$qs2m <- 0.622*((0.6112*exp((17.67*twr$dp4)/(twr$dp4+243.49)))/twr$press)

twr$vPoT43m <- 273.15+twr$PdaT43m*(1+(0.61*twr$qs43m)-0.06)
twr$vPoT37m <- 273.15+twr$PdaT37m*(1+(0.61*twr$qs37m)-0.06)
twr$vPoT8m <- 273.15+twr$PdaT8m*(1+(0.61*twr$qs8m)-0.06)
twr$vPoT2m <- 273.15+twr$PdaT2m*(1+(0.61*twr$qs2m)-0.06)

k=5
twr$rmvPoT43m <- rollmean(twr$vPoT43m, k=k, fill = NA)
twr$rmvPoT37m <- rollmean(twr$vPoT37m, k=k, fill = NA)
twr$rmvPoT8m <- rollmean(twr$vPoT8m, k=k, fill = NA)
twr$rmvPoT2m <- rollmean(twr$vPoT2m, k=k, fill = NA)

# Air for superficial soil temperature

ta <- twr[c("doy","t4")]
ts<- merge(ts,ta,by="doy")
ts <- na.omit(ts)

### Based on Holmes et al 2008 #------------------------------------------------
# The soil temperature was determined at 1 cm of the surface based on TS 5cm
k = 144
z <- -0.05

ts$sa <- runmean(ts$s5cm, k=k, align = "left") # mean tsoil temperatue in a daily basis 24 hours
ts$sec <- (ts$doy-ts$DOY)*86400
ts$sTD <- ts$s5cm-ts$sa
ts$sAmp <- runmax(ts$s5cm, k=k, align = "left")-runmin(ts$s5cm, k=k, align = "left")
w <- (2*pi)/(86400)
TherDiff <- 1.58/(760*837.36)
D <- sqrt(2*TherDiff/w)
ts$s0cm <- ts$sa+(ts$sAmp*exp(-z/D)*sin(w*ts$sec-(z/D)+3.65))

ts <- ts[c("doy","s0cm")]

twr <- merge(twr,ts,by="doy",all = T)
twr <- twr[!(twr$doy<25),];twr <- twr[!(twr$doy>85.35417),]
twr$s0cm <- na.approx(twr$s0cm)

twr$condition <- NA
twr$condition <- ifelse(twr$vPoT43m>(twr$s0cm+273.15),0,ifelse(twr$rain>0,1,0))

#-Lifting Condensation Level----------------------------------------------------
twr$hr<- hour(strptime(twr$date,format = "%d-%m-%Y %H:%M"))

twr$LCL43 <- 125*(twr$t2-twr$dp2);twr$LCL43<-ifelse(twr$LCL43>1,twr$LCL43,1)
twr$LCL37 <- 125*(twr$t1-twr$dp1);twr$LCL37<-ifelse(twr$LCL37>1,twr$LCL37,1)
twr$LCL8 <- 125*(twr$t3-twr$dp3);twr$LCL8<-ifelse(twr$LCL8>1,twr$LCL8,1)
twr$LCL2 <- 125*(twr$t4-twr$dp4);twr$LCL2<-ifelse(twr$LCL2>1,twr$LCL2,1)

twr$sta_43 <- ((twr$vPoT43m-273.15)-twr$s0cm)/43
twr$sta_8 <- ((twr$vPoT8m-273.15)-twr$s0cm)/8
twr$sta_2 <- ((twr$vPoT2m-273.15)-twr$s0cm)/2

twr$SKY <- ifelse((twr$LCL8) > 60,0,ifelse(twr$swc>0.44,10,5))
 
#-Multicomparison---------------------------------------------------------------
twr$swc_10min <- 0
twr$swc_10min <- rollmean(twr$swc,15,fill = NA)
twr$d_swc <- 0
twr$d_swc <- twr$swc[-5]-twr$swc

plumes <- twr[c("doy","LCL43","LCL37","LCL8","LCL2","sta_43","sta_8","sta_2","rain","swc","d_swc")]

plumes <- merge(rcrd,plumes,by="doy")
#write.csv(plumes,file = "./plumes.csv")
plumes$zzz <- 1500

pp <- data.frame(aggregate(plumes$rain,list(plumes$condition),FUN=sum))

plms <- twr[twr$rain>0,]
plms <- plms[plms$LCL43<100,]
plms <- plms[plms$sta_2 > -1,]
plms <- plms[plms$sta_2 < 0,]

plot(twr$doy,twr$s0cm,ylim = c(20,30),type = "l")
points(plms$doy,plms$rh2,type = "h",col="red")

twr$h2 <- 2;twr$h8 <- 8;twr$h37 <- 37;twr$h43 <- 43

# Optionally set colours using RColorBrewer
library(RColorBrewer)
cols = brewer.pal(4, "Blues")

# Define colour pallete
pal = colorRampPalette(c("yellow","red"))

# Rank variable for colour assignment
plot(twr$doy,twr$h2,col=pal(nrow(twr))[twr$order],xlim=c(80,85),ylim = c(0,45))
points(plms$doy,plms$rh2,type = "h",col="grey")
twr$order = findInterval(twr$qs43m, sort(twr$qs43m))
points(twr$doy,twr$h43,col=pal(nrow(twr))[twr$order],type = "h")
twr$order = findInterval(twr$qs43m, sort(twr$qs37m))
points(twr$doy,twr$h37,col=pal(nrow(twr))[twr$order],type = "h")
twr$order = findInterval(twr$qs43m, sort(twr$qs8m))
points(twr$doy,twr$h8,col=pal(nrow(twr))[twr$order],type = "h")
twr$order = findInterval(twr$qs43m, sort(twr$qs2m))
points(twr$doy,twr$h2,col=pal(nrow(twr))[twr$order],type = "h")

plot(ELS$doy,ELS$E_43m,type = "h",xlim=c(80,85),lwd=2)
points(plms$doy,plms$rh2,type = "h",col="grey")
points(ELS$doy,ELS$E_43m,type = "h")
par(new=T)
plot(twr$doy,twr$qs43m,xlim=c(80,85),type="n",col="red",ylim = c(0.01,0.025))

points(twr$doy,twr$qs43m,type = "l",col="red")
points(twr$doy,twr$qs8m,type = "l",col="forestgreen")
points(twr$doy,twr$qs2m,type = "l",col="blue")

## Plots
setwd(b)

#Create a function to generate a continuous color palette
rbPal <- colorRampPalette(c("red","yellow","blue"))
#This adds a column of color values based on the y values
twr$Col <- rbPal(8)[as.numeric(cut(twr$swc,breaks = 8))]

plot(twr$sta_43,twr$LCL43,col=twr$Col,pch=16,cex=0.05,log = "y")
legend("topleft",title="Decile",legend=c(1:10),col =rbPal(10),pch=20)

hist(twr$swc[(twr$sta_8 <0)&(twr$LCL8<100)])

par(mar=c(0.25,5,0.35,0.25),xpd=T) #(DLUR)
layout(matrix(c(1:3),3,1, byrow = F), widths=c(1), heights=c(1))
i=-1;j=1;k=1;l=3000

plot(twr$sta_8,twr$LCL43,ylim = c(k,l),type = "n",xaxt="n",yaxt="n",xlim = c(i,j),log="y",ylab="\n",xlab="\n")
clip(i,j,k,l)
points(twr$sta_8,twr$LCL43,col=twr$Col,pch=1,cex=0.5)

abline(h=60,col="forestgreen")

pdf("plumes_boxplot.pdf", width =9, height = 10)
par(mar=c(4,5,0.75,0.25),xpd=T) #(DLUR)
layout(matrix(c(1:8),4,2, byrow = F), widths=c(1), heights=c(1,1,1,1))
boxplot(plumes$sta_43~plumes$condition,ylim=c(-0.2,0.2),border="red");legend("topright",legend = c(expression("Temperature gradient at 43 m")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression(degree*"C m "^{-1}))
boxplot(plumes$sta_8~plumes$condition,ylimc=c(-1,1),border="forestgreen");legend("topright",legend = c(expression("Temperature gradient at 8 m")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression(degree*"C m "^{-1}))
boxplot(plumes$sta_2~plumes$condition,ylim=c(-2.5,2.5),border="blue");legend("topright",legend = c(expression("Temperature gradient at 2 m")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression(degree*"C m "^{-1}))
boxplot(plumes$swc~plumes$condition,ylim=c(0.4,0.5));legend("topright",legend = c(expression("Soil Moisture ("*Theta*")    ")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression("m"^{3}*" m "^{-3}))
boxplot(plumes$LCL43~plumes$condition,log="y",ylim=c(01,10000),border="red");legend("topright",legend = c(expression("Lifting Condensation Layer at 43 m ("*italic(Z)[lcl]*" )")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression("m"))
boxplot(plumes$LCL8~plumes$condition,log="y",ylim=c(01,10000),border="forestgreen");legend("topright",legend = c(expression("Lifting Condensation Layer at 8 m ("*italic(Z)[lcl]*" )")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression("m"))
boxplot(plumes$LCL2~plumes$condition,log="y",ylim=c(01,10000),border="blue");legend("topright",legend = c(expression("Lifting Condensation Layer at 2 m ("*italic(Z)[lcl]*" )")),bty="n",cex=1.5);mtext(side = 2, line = 2.5, expression("m"))
barplot(pp$x,ylim = c(0,30),names.arg=pp$Group.1);box(lty=1);mtext(side = 2, line = 2.5, expression("mm"));legend("topright",legend = c(expression("Accumulated Precipitation ("*italic(P)*")")),bty="n",cex=1.5)
dev.off()

pdf("detailedmeteo.pdf", width =10, height = 12)
par(mar=c(0.25,5,0.35,0.25),xpd=T) #(DLUR)
layout(matrix(c(1:6),6,1, byrow = F), widths=c(1), heights=c(1,1,1,1,1,1.2))
plot(twr$doy,twr$t2,type = "l",col="red",lwd=0.5,ylim = c(15,35),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
points(twr$doy,twr$dp2,col="indianred1",type = "l",lty=3)
axis(side = 2, at = c(15,20,25,30,35));mtext(side = 2, line = 2.5, expression(italic(T)*" ("*degree*"C)"))
legend("topleft",legend = "43 m",bty="n")
legend("bottomleft",legend = c("Air Temperature","Dew Point"),bty="n",horiz = T,lty = c(1,3),col = c("red","indianred1"))

plot(twr$doy,twr$t3,type = "l",col="forestgreen",lwd=0.5,ylim = c(15,35),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
points(twr$doy,twr$dp3,col="green",type = "l",lty=3)
axis(side = 2, at = c(15,20,25,30,35));mtext(side = 2, line = 2.5, expression(italic(T)*" ("*degree*"C)"))
legend("topleft",legend = "8 m",bty="n")
legend("bottomleft",legend = c("Air Temperature","Dew Point"),bty="n",horiz = T,lty = c(1,3),col = c("forestgreen","green"))

plot(twr$doy,twr$t4,type = "l",col="dodgerblue4",lwd=0.5,ylim = c(15,35),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
points(twr$doy,twr$dp4,col="dodgerblue1",type = "l",lty=3)
axis(side = 2, at = c(15,20,25,30,35));mtext(side = 2, line = 2.5, expression(italic(T)*" ("*degree*"C)"))
legend("topleft",legend = "2 m",bty="n")
legend("bottomleft",legend = c("Air Temperature","Dew Point"),bty="n",horiz = T,lty = c(1,3),col = c("dodgerblue4","dodgerblue"))

plot(twr$doy,twr$s0cm,type = "l",col="brown",lwd=0.5,ylim = c(15,35),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
axis(side = 2, at = c(15,20,25,30,35));mtext(side = 2, line = 2.5, expression(italic(T)*" ("*degree*"C)"))
legend("topleft",legend = "Superficial Soil Temperature",bty="n")

plot(twr$doy,twr$swc,type = "l",col="darkslategrey",lwd=0.5,ylim = c(0.4,0.5),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
axis(side = 2, at = c(0.4,0.45,0.5));mtext(side = 2, line = 2.5, expression(Theta*" (m"^{3}*"m"^{-3}*")"))
legend("topleft",legend = "Soil Moisture",bty="n")

par(mar=c(3,5,0.35,0.25),xpd=T) #(DLUR)
plot(twr$doy,twr$rain,type = "h",col="darkblue",lwd=0.5,ylim = c(0,2.5),yaxt="n",xaxt="n",ylab = "\n",xlab="\n")
axis(side = 2, at = c(0,0.5,1,1.5,2,2.5));mtext(side = 2, line = 2.5, expression(italic(P)*" (mm min"^{-1}*")"))
legend("topleft",legend = "Precipitation",bty="n")
axis(side = 2, at = c(0,0.5,1,1.5,2,2.5))
axis(c("2018-01-25","2018-01-31","2018-02-15","2018-02-28","2018-03-15","2018-03-25"),side = 1, at = c(25,31,46,59,74,84))
dev.off()

pdf("LCL-sampling.pdf", width = 10, height = 12)
layout(matrix(c(1:6),6,1, byrow = T), widths=c(1), heights=c(1,1,1,1,0.75,0.25))

par(mar=c(2.5,6,0,5),xpd=T,font.axis = 1) #(DLUR)
i=80;j=85;k=0;l=1.5
plot(ELS$doy,ELS$E_43m*4, xlim = c(i,j),col="darkgrey",ylim = c(k,l), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(italic(E)*" (mm hr "^{-1}*")"))
axis(side = 2, at = c(0,0.5,1.0,1.5))
axis(c("2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(80,81,82,83,84,85))
clip(i,j,k,l)
z <- "Clear View";g="bisque2";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Plumes";g="cornflowerblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Mist";g="lightblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
points(ELS$doy,ELS$E_43m*4,col="red", type = "h",lwd=1.5,lend=2)
points(ELS$doy,ELS$E_8m*4,col="forestgreen", type = "h",lwd=1.5,lend=2)
points(ELS$doy,ELS$E_2m*4,col="blue", type = "h",lwd=1.5,lend=2)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=1,lty=2,lend=2)

par(mar=c(2.5,5,0,5),xpd=T,font.axis = 1) #(DLUR)
i=80;j=85;k=18;l=32
plot(twr$doy,twr$t1, xlim = c(i,j),col="darkgrey",ylim = c(k,l), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(theta["v"]*" ("*degree*"C)"))
axis(side = 2, at = c(20,25,30))
axis(c("2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(80,81,82,83,84,85))
clip(i,j,k,l)
z <- "Clear View";g="bisque2";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Plumes";g="cornflowerblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Mist";g="lightblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")

points(twr$doy,twr$vPoT43m-273.15,col="red", type = "l",lwd=1)
points(twr$doy,twr$vPoT8m-273.15,col="forestgreen", type = "l",lwd=1)
points(twr$doy,twr$vPoT2m-273.15,col="blue",type = "l",lwd=1)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=1,lty=2)

par(mar=c(2.5,5,0,5),xpd=T,font.axis = 1) #(DLUR)
k=1;l=1500
plot(twr$doy,twr$LCL43,ylim = c(k,l),type = "n",xaxt="n",yaxt="n",xlim = c(i,j),log="y",ylab="\n",xlab="\n")
axis(c("2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(80,81,82,83,84,85))
clip(i,j,k,l)
z <- "Clear View";g="bisque2";points(plumes$doy[plumes$condition==z],plumes$zzz[plumes$condition==z],col=g,type="h")
z <- "Plumes";g="cornflowerblue";points(plumes$doy[plumes$condition==z],plumes$zzz[plumes$condition==z],col=g,type="h")
z <- "Mist";g="lightblue";points(plumes$doy[plumes$condition==z],plumes$zzz[plumes$condition==z],col=g,type="h")

points(twr$doy,twr$LCL43,col="red",type = "l")
points(twr$doy,twr$LCL8,col="forestgreen",type = "l")
points(twr$doy,twr$LCL2,col="blue",type = "l")
abline(h=60,col="green",lty=3,lwd=3,lend=2)
mtext(side = 2, line = 3, expression(italic(z)[lcl]*" (m)"))
axis(side = 2, at = c(1,2,5,10,25,50,100,250,1000,1500))

par(mar=c(2.5,5,0,5),xpd=T,font.axis = 1) #(DLUR)
i=80;j=85;k=18;l=32
plot(c(i,j),c(k,l),type = "n",xaxt="n",yaxt="n",ylab = "\n",xlab = "\n")
clip(i,j,k,l)
z <- "Clear View";g="bisque2";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Plumes";g="cornflowerblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Mist";g="lightblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")

par(new=T)
k=-1.5;l=2.5
plot(twr$doy,twr$sta_43,type = "n",ylim = c(k,l),xlim = c(i,j),xaxt="n",yaxt="n",ylab="\n",xlab="\n")
axis(c("2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(80,81,82,83,84,85))
clip(i,j,k,l)
points(twr$doy,twr$sta_2,type = "h",lend=2,col="blue")
points(twr$doy,twr$sta_8,type = "h",lend=2,col="forestgreen")
points(twr$doy,twr$sta_43,type = "h",lend=2,col="red")
mtext(side = 2, line = 2, expression(over(Delta*theta[v],Delta*"z")))
axis(side = 2, at = c(-1.5,0.5,0,0.5,1.5,2.5))

plot(twr$doy,twr$swc,type = "n",xlim = c(i,j),yaxt="n",xaxt="n",ylab = "\n",ylim = c(0.4,0.48))
clip(i,j,0.4,0.48)
z <- "Clear View";g="bisque2";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Plumes";g="cornflowerblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")
z <- "Mist";g="lightblue";points(plumes$doy[plumes$condition==z],plumes$doy[plumes$condition==z],col=g,type="h")

points(twr$doy,twr$swc,type = "l",col="black",lwd=1,lty=2)
axis(side = 2, at = c(0.4,0.42,0.44,0.46,0.48))
mtext(side = 2, line = 2.5, expression(Theta*" (m"^3*"m "^{-3}*")"))
par(new=T)
plot(twr$doy,twr$rain,type = "n",xlim = c(i,j),yaxt="n",xaxt="n",ylab = "\n",ylim = c(0,2.5))
clip(i,j,0,2.5)
points(twr$doy,twr$rain,type = "h",col="deepskyblue2",lwd=1,lend=2)
mtext(side = 4, line = 3, expression(italic(P)*"( mm min"^{-1}*")"))
axis(side = 4, at = c(0,1,2))
axis(c("2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(80,81,82,83,84,85))

par(mar=c(0,0,0,0),xpd=T,font.axis = 1) #(DLUR)
plot.new()
legend("top",legend = c("43m","8m","2m",expression("s.0"),expression("43m"),expression("8m"),expression("2m"),expression(Theta),expression(italic(P))),col=c("red","forestgreen","blue","black","red","forestgreen","blue","darkslateblue","deepskyblue2"),lty=c(1,1,1,2,NA,NA,NA,2,NA),pch=c(NA,NA,NA,NA,15,15,15,NA,15),bty="n",horiz = T,cex=1.25,pt.cex=2)
legend("bottom",legend = c("Clear View","Mist","Plumes","Canopy Height"),col=c("bisque2","lightblue","cornflowerblue","green"),lty=c(NA,NA,NA,2),pch=c(15,15,15,NA),bty="n",horiz = T,cex=1.25,pt.cex=2)
dev.off()

pdf("LCL.pdf", width = 10, height = 12)
e <- c(79.3222,79.6,79.71458,80.2555556,82.2319444,82.5458333) #c(P,C,P,F,F,C)
layout(matrix(c(1:4),4,1, byrow = T), widths=c(1), heights=c(1))

par(mar=c(0.25,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
i=75;j=85.25
plot(twr$doy,twr$t1, xlim = c(i,j),col="darkgrey",ylim = c(18,32), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(italic(T)*" ("*degree*"C)"))
axis(side = 2, at = c(20,25,30))
axis(c("2018-02-14","2018-02-15","2018-02-16","2018-02-17","2018-02-18"),side = 1, at = c(45,46,47,48,49))
clip(i,j,15,35)
#points(twr$doy,twr$condition*305,col="grey72",type = "h",lwd=2,lend=2)
points(twr$doy,twr$vPoT43m-273.15,col="red", type = "l",lwd=1)
points(twr$doy,twr$vPoT37m-273.15,col="orange", type = "l",lwd=1)
points(twr$doy,twr$vPoT8m-273.15,col="green", type = "l",lwd=1)
points(twr$doy,twr$vPoT2m-273.15,col="blue",type = "l",lwd=1)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=1,lty=2)
legend("topleft",legend = c("43 m","37 m","8 m","2 m","Canopy (60 m)"),col=c("red","orange","green","blue","chartreuse4"),lty=c(1,1,1,1,2),lwd=1.5,bty="n",horiz = T)

e <- c(79.3222,79.6,79.71458,80.2555556,82.2319444,82.5458333) #c(P,C,P,F,F,C)
f<-c(24,30,24,21,21,30);points(e,f,pch=0,col="red",cex=1.5)
f<-f+1;text(e,c(f,f,f,f,f,f),labels = c("Plumes","Clear","Plumes","Mist","Mist","Clear"),srt=90)

plot(twr$doy,twr$swc,type = "n",xlim = c(i,j),yaxt="n",xaxt="n",ylab = "\n",ylim = c(0.4,0.48))
clip(i,j,0.3,0.50)
points(twr$doy,twr$swc,type = "l",col="darkslateblue",lwd=1)
axis(side = 2, at = c(0.4,0.42,0.44,0.46,0.48))
mtext(side = 2, line = 2.5, expression(Theta*" (-)"))
par(new=T)
plot(twr$doy,twr$rain,type = "n",xlim = c(i,j),yaxt="n",xaxt="n",ylab = "\n",ylim = c(0,2.5))
clip(i,j,0,2.5)
points(twr$doy,twr$rain,type = "h",col="deepskyblue2",lwd=1,lend=2)
mtext(side = 4, line = 3, expression(italic(P)*"( mm min"^{-1}*")"))
axis(side = 4, at = c(0,1,2))
f<-1;points(e,c(f,f,f,f,f,f),pch=c(2,0,2,1,1,0),col="red",cex=1.5)
f<-1.25;text(e,c(f,f,f,f,f,f),labels = c("Plumes","Clear","Plumes","Mist","Mist","Clear"),srt=90)
legend("topleft",legend = c(expression(Theta),expression(italic(P)),"Clear Ski","Mist","Plumes"),col=c("darkslateblue","deepskyblue2","red","red","red"),lty=c(1,NA,NA,NA,NA),pch=c(NA,15,0,1,2),bty="n",horiz = T)

par(mar=c(0.25,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
i=75;j=85.25;k=1;l=3000
plot(twr$doy,twr$LCL43,ylim = c(k,l),type = "n",xaxt="n",yaxt="n",xlim = c(i,j),log="y",ylab="\n",xlab="\n")
clip(i,j,k,l)
points(twr$doy,twr$LCL43,col="red",type = "l")
points(twr$doy,twr$LCL37,col="orange",type = "l")
points(twr$doy,twr$LCL8,col="green",type = "l")
points(twr$doy,twr$LCL2,col="blue",type = "l")
abline(h=60,col="chartreuse4",lty=2,lwd=3,lend=2)
f<-60;points(e,c(f,f,f,f,f,f),pch=c(2,0,2,1,1,0),col="red",cex=1.5)
mtext(side = 2, line = 3, expression(italic(z)[lcl]*" (m)"))
axis(side = 2, at = c(1,2,5,10,25,50,100,250,1000))
legend("topleft",legend = c("43 m","37 m","8 m","2 m","Canopy (60 m)"),col=c("red","orange","green","blue","chartreuse4"),lty=c(1,1,1,1,2),lwd=1.5,bty="n",horiz = T)

k=-0.5;l=1
par(mar=c(3,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
plot(twr$doy,twr$sta_43,type = "n",ylim = c(k,l),xlim = c(i,j),xaxt="n",yaxt="n",ylab="\n",xlab="\n")
clip(i,j,k,l)
points(twr$doy,twr$sta_8,type = "h",lend=2,col="black")
points(twr$doy,twr$sta_43,type = "h",lend=2,col="darkslategray4")
f<-0;points(e,c(f,f,f,f,f,f),pch=c(2,0,2,1,1,0),col="red",cex=1.5)
mtext(side = 2, line = 3, expression(zeta*" (-)"))
axis(side = 2, at = c(-0.5,0,0.5,1))
axis(c("2018-03-16","2018-03-17","2018-03-18","2018-03-19","2018-03-20","2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(75,76,77,78,79,80,81,82,83,84,85))
legend("topleft",legend = c(expression(iota["43m"]),expression(iota["8m"])),col=c("darkslategray4","black"),pch=c(15,15),bty="n",horiz = T)
dev.off()

# Temperature ##############

#pdf("tempPlumes.pdf", width = 9, height = 8)
layout(matrix(c(1:4),4,1, byrow = T), widths=c(1), heights=c(0.5,1,0.5,1))
par(mar=c(0.25,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
i=82.2;j=82.8
plot(ots$doy,ots$rf*4,type="n",ylim = c(1,0),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
clip(i,j,0,5)
points(twr$doy,twr$rain,col="darkblue",type = "h",lwd=2,lend=2)
mtext(side = 2, line = 3, expression(italic(P)*"(mm min "^{-1}*")"))
axis(side = 2, at = c(0,0.25,0.5,1))

par(new=T)
plot(twr$doy,twr$swc,type="n",ylim = c(0.37,0.55),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
axis(side = 4, at = c(0.4,0.45,0.5))
mtext(side = 4, line = 3, expression(italic(Theta)*" (m"^{3}*"m"^{-3}*")"))
clip(i,j,0,5)
points(twr$doy,twr$swc,type = "l",col="darkred",lwd=1.5)
legend("bottom",legend = c(expression(italic(P)), expression(Theta)),col=c("darkblue","darkred"),lty=c(NA,1),pch=c(15,NA),lwd=2,bty="n",horiz = T,cex = 1.5)

par(mar=c(5,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
plot(twr$doy,twr$t1, xlim = c(i,j),col="darkgrey",ylim = c(15,35), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(italic(T)*" ("*degree*"C)"))
axis(side = 2, at = c(15,20,25,30,35))
mtext(side = 1, line = 3, expression("2018-03-23 (HH:MM)"))
axis(c("06:00","09:00","12:00","15:00","18:00"),side = 1, at = c(82.25,82.375,82.5,82.625,82.75))
clip(i,j,0,400)
points(twr$doy,twr$vPoT43m-273.15,col="red", type = "l",lwd=2)
points(twr$doy,twr$vPoT37m-273.15,col="orange", type = "l",lwd=2)
points(twr$doy,twr$vPoT8m-273.15,col="green", type = "l",lwd=2)
points(twr$doy,twr$vPoT2m-273.15,col="blue",type = "l",lwd=2)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=2,lty=2)
legend("bottom",legend = c(expression(theta["v:43m"]), expression(theta["v:37m"]),expression(theta["v:8m"]),expression(theta["v:2m"]),expression(italic(T)["ss"])),col=c("red","orange","green","blue","black"),lty=c(1,1,1,1,2),lwd=2,bty="n",horiz = T,cex = 1.5)

par(mar=c(0.25,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
i=83.2;j=83.8
plot(ots$doy,ots$rf*4,type="n",ylim = c(1,0),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
clip(i,j,0,5)
points(twr$doy,twr$rain,col="darkblue",type = "h",lwd=2,lend=2)
mtext(side = 2, line = 3, expression(italic(P)*"(mm min "^{-1}*")"))
axis(side = 2, at = c(0,0.25,0.5,1))

par(new=T)
plot(twr$doy,twr$swc,type="n",ylim = c(0.37,0.55),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
axis(side = 4, at = c(0.4,0.45,0.5))
mtext(side = 4, line = 3, expression(italic(Theta)*" (m"^{3}*"m"^{-3}*")"))
clip(i,j,0,5)
points(twr$doy,twr$swc,type = "l",col="darkred",lwd=1.5)
legend("bottom",legend = c(expression(italic(P)), expression(Theta)),col=c("darkblue","darkred"),lty=c(NA,1),pch=c(15,NA),lwd=2,bty="n",horiz = T,cex = 1.5)

par(mar=c(4,5,0.5,5),xpd=T,font.axis = 1) #(DLUR)
plot(twr$doy,273.15+twr$t1, xlim = c(i,j),col="darkgrey",ylim = c(21,24), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(italic(T)*" ("*degree*"C)"))
axis(side = 2, at = c(21,22,23,24))
mtext(side = 1, line = 3, expression("2018-03-24 (HH:MM)"))
axis(c("06:00","09:00","12:00","15:00","18:00"),side = 1, at = c(83.25,83.375,83.5,83.625,83.75))
clip(i,j,0,305)
points(twr$doy,twr$vPoT43m-273.15,col="red", type = "l",lwd=2)
points(twr$doy,twr$vPoT37m-273.15,col="orange", type = "l",lwd=2)
points(twr$doy,twr$vPoT8m-273.15,col="green", type = "l",lwd=2)
points(twr$doy,twr$vPoT2m-273.15,col="blue",type = "l",lwd=2)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=2,lty=2)
legend("bottom",legend = c(expression(theta["v:43m"]), expression(theta["v:37m"]),expression(theta["v:8m"]),expression(theta["v:2m"]),expression(italic(T)["ss"])),col=c("red","orange","green","blue","black"),lty=c(1,1,1,1,2),lwd=2,bty="n",horiz = T,cex = 1.5)
dev.off()


# Week ##############
pdf("week.pdf", width = 12, height = 7)
layout(matrix(c(1:2),2,1, byrow = T), widths=c(1), heights=c(1,0.5))
par(mar=c(0,4,0.2,5),xpd=T,font.axis = 1) #(DLUR)
i=75;j=85.25
plot(twr$doy,twr$t1, xlim = c(i,j),col="darkgrey",ylim = c(18,32), type="n",xaxt="n",yaxt="n", xlab = "\n",ylab = "\n",lwd=2)
mtext(side = 2, line = 3, expression(italic(T)*" ("*degree*"C)"))
axis(side = 2, at = c(20,25,30))
axis(c("2018-02-14","2018-02-15","2018-02-16","2018-02-17","2018-02-18"),side = 1, at = c(45,46,47,48,49))
clip(i,j,15,35)
#points(twr$doy,twr$condition*305,col="grey72",type = "h",lwd=2,lend=2)
points(twr$doy,twr$vPoT43m-273.15,col="red", type = "l",lwd=1)
points(twr$doy,twr$vPoT37m-273.15,col="orange", type = "l",lwd=1)
points(twr$doy,twr$vPoT8m-273.15,col="green", type = "l",lwd=1)
points(twr$doy,twr$vPoT2m-273.15,col="blue",type = "l",lwd=1)
points(ts$doy,ts$s0cm,col="black",type = "l",lwd=1,lty=2)

e <- c(79.3222,79.6,79.71458,80.2555556,82.2319444,82.5458333) #c(P,C,P,F,F,C)
f<-c(24,30,24,21,21,30);points(e,f,pch=0,col="red",cex=1.5)
f<-f+1;text(e,c(f,f,f,f,f,f),labels = c("Plumes","High Clouds","Plumes","Mist","Mist","High Clouds"),srt=90)

par(mar=c(5,4,0,5),xpd=T,font.axis = 1) #(DLUR)
plot(twr$doy,twr$rain,type="n",ylim = c(0,2.5),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
clip(i,j,0,5)
points(twr$doy,twr$rain,col="darkblue",type = "h",lwd=2,lend=2)
mtext(side = 2, line = 3, expression(italic(P)*"(mm min "^{-1}*")"))
axis(side = 2, at = c(0,0.5,1,1.5,2))
par(new=T)
plot(twr$doy,twr$swc,type="n",ylim = c(0.40,0.48),xlim = c(i,j),xaxt="n",yaxt="n",ylab = "\n",xlab = "\n",lwd=1,cex=2,lend=2)
axis(side = 4, at = c(0.4,0.42,0.44,0.46,0.48))
mtext(side = 4, line = 3, expression(italic(Theta)*" (m"^{3}*"m"^{-3}*")"))
clip(i,j,0,5)
points(twr$doy,twr$swc,type = "l",col="darkred",lwd=1.5)
legend("topleft",legend = c(expression(italic(P)), expression(Theta)),col=c("darkblue","darkred"),lty=c(NA,1),pch=c(15,NA),lwd=2,bty="n",horiz = T,cex = 1)
axis(c("2018-03-16","2018-03-17","2018-03-18","2018-03-19","2018-03-20","2018-03-21","2018-03-22","2018-03-23","2018-03-24","2018-03-25","2018-03-26"),side = 1, at = c(75,76,77,78,79,80,81,82,83,84,85))
dev.off()